#!/bin/bash
#
# Copyright 2017 The Android Open Source Project
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

set -e

function dir_with_file() {
  local file=${1}; shift
  local dir;
  for dir; do
    if [[ -z "${dir}" ]]; then continue; fi
    if [[ -r "${dir}/${file}" ]]; then
      echo "${dir}"
      return
    fi
  done
  echo "Could not find ${file}, looked in $@" >&2
  return 1
}

LOCAL_DIR=$(dirname "${0}")

# Environment variable options
#
# By default, make BOARD empty. Picks up MT8516.
BOARD=${BOARD:-""}
# Convert to lower case
BOARD="$(echo ${BOARD} | tr '[A-Z]' '[a-z]')"
# By default, flash both a and b slots. Override to only flash a.
FLASHB=${FLASHB:-true}
# By default, flash-all will reboot the device out of fastboot mode when it is
# done flashing.
REBOOT=${REBOOT:-true}
# By default, flash oem partition. Override to skip flashing oem.
FLASH_OEM=${FLASH_OEM:-true}
# By default, don't flash oem partition containing GMSCore. Override to
# oem_dev.img/vbmeta_dev.img if available.
USE_GMSCORE=${USE_GMSCORE:-false}
# By default, the userdata partition will be reformatted.
WIPE_USER_DATA=${WIPE_USER_DATA:-true}
# By default, flash-all will check if the device is AVB locked and error out if
# so. Override to flash anyway.
FLASH_IF_LOCKED=${FLASH_IF_LOCKED:-false}
# By default, flash the factory partition if an image is present.
FLASH_FACTORY_DATA=${FLASH_FACTORY_DATA:-true}

# Location of where the OS image is built.
OS=$(dir_with_file boot.img \
    "${ANDROID_PROVISION_OS_PARTITIONS}" \
    "${LOCAL_DIR}" \
    "${ANDROID_PRODUCT_OUT}")

# Try to protect user from triggering rollback protection when flashing. Check
# the AVB lock state and only allow flashing if the device is unlocked or if the
# user explicitly overrides using the FLASH_IF_LOCKED environment variable.
set +e
AVB_LOCKED_STR=$(fastboot getvar at-vboot-state "$@" 2>&1 | grep 'avb-locked')
set -e
AVB_LOCKED=$(echo "$AVB_LOCKED_STR" | cut -d' ' -f3)
if [[ -z "$AVB_LOCKED_STR" ]]; then
  if [[ "$FLASH_IF_LOCKED" != "true" ]]; then
    echo "Couldn't determine AVB lock state. Flashing may trigger rollback protection."
    echo "Set 'FLASH_IF_LOCKED=true' to override."
    exit 1
  fi
  echo "Couldn't determine AVB lock state but FLASH_IF_LOCKED is set. Continuing with flashing."
elif [[ "$AVB_LOCKED" == "0" ]]; then
  echo "AVB is unlocked. Flashing."
elif [[ "$FLASH_IF_LOCKED" == "true" ]]; then
  echo "AVB is locked but FLASH_IF_LOCKED is set. Continuing with flashing."
else
  echo "AVB is locked. Flashing may trigger rollback protection."
  echo "Unlock first or set 'FLASH_IF_LOCKED=true' to override."
  exit 1
fi

# Check whether this board still has the older preloader/lk and partition table,
# before the rename to lk_a/lk_b and before proper partition table support in
# the bootloader. If so, we need to update it to the newer one with a modified
# flash procedure. We query the 'lk' partition size to determine if the table is
# old. Size will be non-empty if the partition exists:
#   [old] partition-size:lk: 60000
#   [new] partition-size:lk:
exec 5>&1
GETVAR_SIZE=$(fastboot getvar partition-size:lk "$@" 2>&1 | tee >(cat - >&5))
LK_PART_SIZE=$(echo "$GETVAR_SIZE" | grep 'partition-size' | cut -d' ' -f2)
exec 5>&-
if [[ "$LK_PART_SIZE" =~ ^(0x)?[0-9A-Fa-f]+$ ]]; then
  echo "Detected old partition table, updating..."
  ARGS=(
    flash boot0     "${OS}"/boot0.img
    flash lk        "${OS}"/lk.img
    flash lk2       "${OS}"/lk.img
    flash tee1      "${OS}"/tee.img
    flash tee2      "${OS}"/tee.img
    flash partition "${OS}"/partition-table.img
    reboot bootloader
  )
  fastboot "${ARGS[@]}" "$@"
  sleep 5
fi

# Minimal flash commands needed to flash potentially changed partition table and
# reboot back into the bootloader.
# Erase the misc partition to clear out any stale state, such as whether to
# reboot into recovery
ARGS=(
  flash boot0     "${OS}"/boot0.img
  flash partition "${OS}"/partition-table.img
  flash tee_a     "${OS}"/tee.img
  flash lk_a      "${OS}"/lk.img
  erase misc
  set_active a
)

if [[ "${FLASHB}" = true ]]; then
  ARGS+=(
    flash tee_b "${OS}"/tee.img
    flash lk_b  "${OS}"/lk.img
  )
fi

# The bootloader reloads the new partition table immediately without needing to
# reboot.
fastboot "${ARGS[@]}" "$@"

# Build the remainder of the fastboot command
ARGS=(
  flash system_a "${OS}"/system.img
  flash boot_a   "${OS}"/boot.img
  flash vendor_a "${OS}"/vendor.img
  flash logo     "${OS}"/logo.img
)

if [[ "${FLASHB}" = true ]]; then
  ARGS+=(
    flash system_b "${OS}"/system.img
    flash boot_b   "${OS}"/boot.img
    flash vendor_b "${OS}"/vendor.img
  )
fi

if [[ "${FLASH_OEM}" = true ]]; then

  # Use board specific version for others
  # File does not exist for unsigned images
  if [[ ! -z "${BOARD}" ]] && [[ -f "${OS}"/vbmeta_${BOARD}.img ]]; then
    VBMETA_IMAGE=vbmeta_${BOARD}.img
  else
    VBMETA_IMAGE=vbmeta.img
  fi

  if [[ ! -z "${BOARD}" ]] && [[ -f "${OS}"/oem_${BOARD}.img ]]; then
    OEM_IMAGE=oem_${BOARD}.img
  else
    OEM_IMAGE=oem.img
  fi

  if [[ "${USE_GMSCORE}" = true && -f "${OS}"/oem_dev.img && \
        -f "${OS}"/vbmeta_dev.img ]]; then
    OEM_IMAGE=oem_dev.img
    VBMETA_IMAGE=vbmeta_dev.img
  fi
  ARGS+=(
    flash oem_a     "${OS}"/"${OEM_IMAGE}"
    flash vbmeta_a  "${OS}"/"${VBMETA_IMAGE}"
  )
  if [[ "${FLASHB}" = true ]]; then
    ARGS+=(
      flash oem_b     "${OS}"/"${OEM_IMAGE}"
      flash vbmeta_b  "${OS}"/"${VBMETA_IMAGE}"
    )
  fi
fi

if [[ ! -z "${BOARD}" ]] && [[ -f "${OS}"/oem_bootloader_${BOARD}.img ]]; then
  OEM_BOOTLOADER_IMAGE=oem_bootloader_${BOARD}.img
elif [[ -f "${OS}"/oem_bootloader.img ]]; then
  OEM_BOOTLOADER_IMAGE=oem_bootloader.img
else
  OEM_BOOTLOADER_IMAGE=""
fi

if [[ ! -z "${OEM_BOOTLOADER_IMAGE}" ]]; then
  ARGS+=( flash oem_bootloader_a "${OS}"/"${OEM_BOOTLOADER_IMAGE}" )
  if [[ "${FLASHB}" = true ]]; then
    ARGS+=( flash oem_bootloader_b "${OS}"/"${OEM_BOOTLOADER_IMAGE}" )
  fi
fi

# Flash the /factory data partition if present and desired.
if [[ "${FLASH_FACTORY_DATA}" = true && -f "${OS}"/factory.img ]]; then
  ARGS+=( flash factory "${OS}"/factory.img )
fi

fastboot "${ARGS[@]}" "$@"

# Must reboot before trying to format userdata (b/66924340)
if [[ "${WIPE_USER_DATA}" = true ]]; then
  fastboot reboot bootloader "$@"
  sleep 5

  fastboot format userdata "$@"
fi

if [[ "${REBOOT}" = true ]]; then
  fastboot reboot "$@"
else
  echo "Skipping reboot and staying in fastboot mode."
fi

echo
echo "Successfully flashed your mt8516."
